﻿using Microsoft.Crm.Sdk.Messages;
using Microsoft.Xrm.Sdk;
using System;
using System.Collections.Generic;
using System.Linq;
using System.Threading.Tasks;
using VA.PPMS.Context;
using VA.PPMS.Context.Interface;
using VA.PPMS.IWS.Common;
using VA.PPMS.ProviderData;

namespace VA.PPMS.IWS.MappingService.Mappers
{
    public class MapCcnIdentifierToCrm : MapperRelatedBase
    {
        public MapCcnIdentifierToCrm(IPpmsContextHelper ppmsContextHelper, IPpmsHelper ppmsHelper)
            : base(ppmsContextHelper, ppmsHelper)
        {}

        public async Task<ppms_providernetworkid> MapInsert(ProviderNetworkId schemaId, Account provider)
        {
            if (!schemaId.NetworkAsGuid.HasValue) return null;

            var id = new ppms_providernetworkid
            {
                Id = Guid.NewGuid(),
                ppms_providerid = schemaId.Number,
                ppms_network = new EntityReference("ppms_vaprovidernetwork", schemaId.NetworkAsGuid.Value )
            };

            await Task.Run(() => {});

            return id;
        }

        public async Task<ppms_providernetworkid> MapUpdate(ProviderNetworkId id, Account provider)
        {
            // Check for network ID
            if (!id.NetworkAsGuid.HasValue) return null;

            // check if entity exists
            var crmEntity = GetCrmEntity(provider, id.NetworkId);
            if (crmEntity == null)
            {
                return await MapInsert(id, provider);
            }

            // check for matching network
            if (id.NetworkAsGuid.Value != crmEntity.ppms_network.Id)
            {
                throw new PpmsServiceException("Network ID mismatch.");
            }

            var entity = new ppms_providernetworkid()
            {
                Id = crmEntity.Id,
                ppms_providerid = crmEntity.ppms_providerid
            };

            // name
            if (IsChanged(id.Number, crmEntity.ppms_providerid))
            {
                entity.ppms_providerid = id.Number;
            }

            // return mapped entity
            return entity;
        }

        private static ppms_providernetworkid GetCrmEntity(Account provider, string id)
        {
            var list = provider.ppms_account_providernetworkid;
            if (list == null) return null;

            var ppmsProvidernetworkids = list as ppms_providernetworkid[] ?? list.ToArray();

            return ppmsProvidernetworkids.Any() ? ppmsProvidernetworkids.FirstOrDefault(x => x.ppms_providerid == id) : null;
        }

        private static ProviderNetworkId ConvertEntity<T>(T entity)
        {
            return (ProviderNetworkId)Convert.ChangeType(entity, typeof(ProviderNetworkId));
        }

        public override async Task<Entity> MapInsert<T>(T entity, Entity parent)
        {
            return await MapInsert(ConvertEntity(entity), (Account)parent);
        }

        public override async Task<Entity> MapUpdate<T>(T entity, Entity parent)
        {
            return await MapUpdate(ConvertEntity(entity), (Account)parent);
        }

        public override void AddChildrenToProvider(IList<Entity> entities, Entity parent)
        {
            if (IsWithinContext) AssociateRelatedEntities(parent, entities, "ppms_account_providernetworkid");
            else
            {
                if (entities != null && entities.Count > 0)
                {
                    var account = (Account)parent;
                    if (account != null)
                        account.ppms_account_providernetworkid = ConvertEntityList<ppms_providernetworkid>(entities);
                }
            }
        }

        public override IEnumerable<SetStateRequest> MapDelete<T>(IList<T> entities, Entity parent)
        {
            if (entities == null || !entities.Any()) return null;

            // Check provider
            var provider = (Account)parent;
            if (provider == null) return null;

            var list = new List<ppms_providernetworkid>();

            // Map schema entities for delete
            foreach (var item in entities)
            {
                var entity = ConvertEntity(item);
                var matches = provider.ppms_account_providernetworkid.Where(p => p.ppms_providerid == entity.Number);
                list.AddRange(matches);
            }

            return EntityDelete((IEnumerable<ppms_providernetworkid>)list);
        }
        
    }
}